/*

	Copyright (C) 2006  Dave Keck

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA

*/

#import "Controller.h"

#import "aw_lib.h"

/* Comment-out this next line to use the callback scenario. */

//#define BUFFER_EVENTS

/* Global Variables */

static NSTableView *gPinValuesTable = NULL;
static unsigned char readValues[16];

/* Our data-changed callback. */

void boardConnected(AW_EVENT event);
void boardDisconnected(AW_EVENT event);
void boardDataChanged(AW_EVENT event);

@implementation Controller

- (void)awakeFromNib
{

    AW_EVENT_DELIV_METHOD eventDeliveryMethod = AW_EVENT_DELIV_CALLBACK;
	unsigned char tempData[2] = {0, 0},
				  numberOfBoards = 0;
	AW_ERR result;
	
	/* Setup our global version of pinValuesTable to be accessed from our callback, boardDataChanged() */
	
	gPinValuesTable = pinValuesTable;
	
	/* Get everything ready. */
    
	result = aw_init();
	
		if (result) [self bail: @"Unable to open the ActiveWire board. Exiting..."];
	
	/* Verifies we have a connected board. */
	
	result = aw_numberOfConnectedBoards(&numberOfBoards);
	
		if (result || !numberOfBoards) [self bail: @"No boards currently connected. Exiting..."];
	
	/* Sets all the pins to output. */
	
	result = aw_setDirectionsOfPinSets(0, 255, 255);
	
		if (result) [self bail: @"Unable to set the IO pins' directions. Exiting..."];
    
    /* Get arrays ready... */
    
    bzero(pinValues, 16);
    
    bzero(readValues, 16);
    
    /* This actually turns the pins off physically. */
    
    [self turnOffAllPins];
	
	result = aw_readData(0, tempData, 2);
	
		if (result) [self bail: @"Unable to read data from ActiveWire board. Exiting..."];
	
	aw_decimalToBinary(tempData, 2, readValues);
    
	/* Updates the window's representation of the pins. */
	
    [pinValuesTable reloadData];
	
	#ifdef BUFFER_EVENTS
	
		eventDeliveryMethod = AW_EVENT_DELIV_BUFFER;
		
		[NSTimer scheduledTimerWithTimeInterval: 0.25 target: self selector: @selector(checkForNewEvents) userInfo: nil repeats: YES];
	
	#endif
	
	result = aw_setEventDeliveryMethod(eventDeliveryMethod, boardConnected, boardDisconnected, boardDataChanged, 0);
	
		if (result) [self bail: @"Unable to set event-delivery method. Exiting..."];

}

- (void)dealloc
{

    [self turnOffAllPins];
    
	[self bail: @"Exiting..."];
	
    [super dealloc];

}

- (IBAction)setPinValue: (id)sender
{

    if ([sender state])
		pinValues[[sender tag]] = 1;
    
    else
		pinValues[[sender tag]] = 0;
    
	[self updatePins];

}

- (void)updatePins
{

    unsigned char tempData[2];
	
	aw_binaryToDecimal(pinValues, 2, tempData);
	
	if (aw_writeData(0, tempData, 2))
		NSLog(@"Error: couldn't write data.");

}

- (void)turnOffAllPins
{

    unsigned char data[2] = {0, 0};
    
	if (aw_writeData(0, data, 2))
		NSLog(@"Error: couldn't write data.");

}

/* Only used in the buffered-events scenerio. */

- (void)checkForNewEvents
{

	AW_EVENT_LIST eventList = aw_getEvents();
	AW_EVENT currentEvent = NULL;
	int i;
	unsigned char binaryArray[16];
	
		if (!eventList->numberOfEvents) return;
	
	for (i = 0; i <= eventList->numberOfEvents - 1; i++)
	{
	
		currentEvent = eventList->events[i];
		
		NSLog(@"Board #%d data changed at %@!", currentEvent->boardNumber,
			[[NSDate dateWithTimeIntervalSince1970: (NSTimeInterval)currentEvent->timestamp.tv_sec] description]);
		
		/* We only treat the data of the board at the time of the event as current
		   data (and storing it in readValues) if it's the last registered event. */
		
		if (i == eventList->numberOfEvents - 1)
		{
		
			aw_decimalToBinary(currentEvent->boardData, 2, readValues);
			
			[pinValuesTable reloadData];
		
		}
		
		else
		{
		
			aw_decimalToBinary(currentEvent->boardData, 2, binaryArray);
			
			NSLog(@"Board data at time: %d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d.", binaryArray[0], binaryArray[1],
																			binaryArray[2], binaryArray[3],
																			binaryArray[4], binaryArray[5],
																			binaryArray[6], binaryArray[7],
																			binaryArray[8], binaryArray[9],
																			binaryArray[10], binaryArray[11],
																			binaryArray[12], binaryArray[13],
																			binaryArray[14], binaryArray[15]);
		
		}
	
	}
	
	/* We're finished with these events so we can clear them from memory. */
	
	aw_clearEvents();

}

- (void)bail: (NSString *)reason
{

	NSLog(reason);
	
	/* We're feelin courteous so we'll say our graceful goodbyes
	   by resetting the board's pins before storming out. */
	
	aw_setDirectionsOfPinSets(0, 0, 0);
	
	aw_close();
	
	exit(1);

}

#pragma mark -
#pragma mark NSTableDataSource Methods
#pragma mark -

- (int)numberOfRowsInTableView: (NSTableView *)aTableView
{

    return 16;

}

- (id)tableView: (NSTableView *)aTableView objectValueForTableColumn: (NSTableColumn *)aTableColumn row: (int)rowIndex
{

    if ([[aTableColumn identifier] isEqualToString: @"0"])
		return [NSString stringWithFormat: @"%d", rowIndex];
    
    if (readValues[rowIndex])
		return @"On";
    
    return @"Off";

}

@end

#pragma mark -
#pragma mark Callbacks
#pragma mark -

void boardConnected(AW_EVENT event)
{

	NSLog(@"### Board #%d connected!", event->boardNumber);
	
	/* Sets all the pins to output. */
	
	aw_setDirectionsOfPinSets(event->boardNumber, 255, 255);

}

void boardDisconnected(AW_EVENT event)
{

	NSLog(@"### Board #%d disconnected!", event->boardNumber);

}

void boardDataChanged(AW_EVENT event)
{

	NSLog(@"### Board #%d data changed!", event->boardNumber);
	
	aw_decimalToBinary(event->boardData, 2, readValues);
	
	NSLog(@"Latest board data: %d%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d.", readValues[0], readValues[1],
																   readValues[2], readValues[3],
																   readValues[4], readValues[5],
																   readValues[6], readValues[7],
																   readValues[8], readValues[9],
																   readValues[10], readValues[11],
																   readValues[12], readValues[13],
																   readValues[14], readValues[15]);
	
	[gPinValuesTable performSelectorOnMainThread: @selector(reloadData) withObject: nil waitUntilDone: YES];

}